/*
________________________________________________________________________________

   04.make-numerator.sas 
   OPA Measure Calculations
   Step 4: Define numerator for measures

   Summary  : Defines numerator using dataset created in 03.denominator.sas
   program, to find patients with LARC and Most/Mod claims. 

   Authors  : P. Hastings / phil -at- farharbor -dot- com
            : F. Dong / fei -at- farharbor -dot- com 
			: E. Puga / ella -at- farharbor -dot- com   
            : Based on original code by B. Frederiksen -at- DHHS/OPA
   Version  : v3.01
   Date     : 2020-03-20
   Revisions: None.
   History  : 2016-12-19(1.00), 2017-02-22(1.01), 2017-05-17(1.02), 
              2018-04-20(1.03), 2018-08-20(1.03a), 2019-02-26 (2.01)
________________________________________________________________________________

*/

*=============================================================================;
*  CREATE NUMERATOR FOR THE MEASURES HERE, MEM/LARC ALGORITHM
*=============================================================================;

*===;
*     RECODE/FLAG VARIOUS LARC CONDITIONS - ALL WOMEN;
*===;

*...This creates a dataset for Methods;
data Method; 
   set NumeratorData; 
   run;   

*...Sort Method data by Patient ID and date of service;
proc sort data = Method; 
   by PatientID_OPA DOSNew; 
   run;

*...This section counts the LARC even if removed, but uses original LARC definition (FD: excludes surv codes for IUD/Implant);
*...Use retain statement to flag LARCs [PH];
data LARCFlag (keep = PatientID_OPA Flag_HadLARC_new);
   set Method; 
   retain Flag_HadLARC_new;
   by PatientID_OPA DOSNew;
   if first.PatientID_OPA then Flag_HadLARC_new = 0;
   if MethodLARC = 1 then Flag_HadLARC_new = 1;
   if last.PatientID_OPA then output;
   run;

*...FD EDIT === UPDATE Feb 2019 === ;
*...FD: add a method flag for each individual method (all include surv codes), and keep them in the final dataset;
*...Use retain statement to flag methods;
data MethodFlag (keep = PatientID_OPA Flag_FemSteril Flag_IUD Flag_Implant Flag_Inject Flag_OC Flag_Patch Flag_Ring Flag_Diaphragm);
   set Method;
   retain Flag_FemSteril Flag_IUD Flag_Implant Flag_Inject Flag_OC Flag_Patch Flag_Ring Flag_Diaphragm;
   by PatientID_OPA DOSNew;
   if first.PatientID_OPA then do;
   Flag_FemSteril = 0;
   Flag_IUD = 0;
   Flag_Implant = 0;
   Flag_Inject = 0;
   Flag_OC = 0;
   Flag_Patch = 0;
   Flag_Ring = 0;
   Flag_Diaphragm = 0; end;
   if FemSteril = 1 then Flag_FemSteril = 1;
   if IUD = 1 then Flag_IUD = 1;
   if Implant = 1 then Flag_Implant = 1;
   if Inject = 1 then Flag_Inject = 1;
   if OC = 1 then Flag_OC = 1;
   if Patch = 1 then Flag_Patch = 1;
   if Ring = 1 then Flag_Ring = 1;
   if Diaphragm = 1 then Flag_Diaphragm = 1;
   if last.PatientID_OPA then output;
   run;


*...Create MergeMethodLARC dataset by merging Method dataset and LARCFlag dataset by Patient ID;
data MergeMethodLARC (keep = PatientID_OPA DOSNew Method MethodEffect MethodMost Flag_HadLARC_new Flag_FemSteril Flag_IUD Flag_Implant Flag_Inject Flag_OC Flag_Patch Flag_Ring Flag_Diaphragm); 
   merge method (in=a) LARCFlag MethodFlag;  *...FD: add MethodFlag and keep the flags;
   by PatientID_OPA;
   if a=1;
   run;

*...Sort MergeMethodLARC dataset by Patient ID and MethodEffect to create a dataset with the last record for 
    each individual being her most effective method in the measurement year;
proc sort data = MergeMethodLARC;
   by PatientID_OPA MethodEffect;
   run;

*...Create a one record per person dataset with the most effective method in the measurement year variables;
data ContraceptiveDataMEM (keep = PatientID_OPA MEM_Method_new MEM_MethodEffect_new MEM_MethodMost_new Flag_HadLARC_new 
                           Flag_FemSteril Flag_IUD Flag_Implant Flag_Inject Flag_OC Flag_Patch Flag_Ring Flag_Diaphragm); 
   set MergeMethodLARC; 
   by PatientID_OPA MethodEffect;
   if last.PatientID_OPA;
   MEM_Method_new = Method;
   MEM_MethodEffect_new = MethodEffect;
   MEM_MethodMost_new = MethodMost;
   run;

*...Sort ContraceptiveDataMEM dataset by Patient ID to prepare to merge with MergeMethodLARC dataset to 
    create a final dataset with most effective method in the measurement year variables;
proc sort data = ContraceptiveDataMEM; 
   by PatientID_OPA;
   run;

*...Sort by Patient ID to prepare to merge with Method dataset to create final MEM dataset;
proc sort data = Method; 
   by PatientID_OPA;
   run;

*...Create MEMLast dataset by merging the MergeMethodLARC dataset with ContraceptiveDataMEM dataset by Patient ID;
data MEM; 
   merge Method
         ContraceptiveDataMEM ;
   by PatientID_OPA;
   run;

*...Delete intermediate datasets;
proc datasets lib = work nolist;
   delete MergeMethodLARC ContraceptiveDataMEM LARCFlag MethodFlag;
   quit;
   run;

*...Sort MEM dataset by Patient ID and first date of service to prepare to create a one record per person dataset;
proc sort data = MEM;
   by PatientID_OPA DOSNew;
   run;

*...Create a one record per person dataset from MEM dataset;
data FinalContraceptiveData_all; 
   set MEM; 
   by PatientID_OPA DOSNew;
   if last.PatientID_OPA;
   run;

*...Delete intermediate datasets;
proc datasets lib = work nolist;
   delete Method MEM NumeratorData NumeratorDataOne;
   quit;
   run;

*===============================;
*     POSTPARTUM SECTION        ;
*===============================;

*...This macro performs numerator operations for postpartum populations and creates separate datasets for each 
    postpartum time period (triggered by the postpartum=Y option in 00.setup.sas);

%macro numerator_pp_new;
%if &postpartum_st. = y %then 
   %do;
   %let i = 1;
      %do %while (%scan(&dayspp_list, &i) ne );
      %let n = %scan(&dayspp_list, &i);
      
      %*...This creates a dataset (Method) with only observations where a method could be defined from the codes above making 
           a smaller dataset to work with;
      data Method_pp&n;
         set NumeratorData_pp;
         where flag_pp&n. = 1;
         run;

      %*...Sort Method data by Patient ID and date of service;
      proc sort data = Method_pp&n; 
         by PatientID_OPA DOSNew;
         run;

      %*...This section counts the LARC even if removed, but uses original LARC definition (FD: excludes surv codes for IUD/Implant)
           and uses a retain statement to flag LARC;
      data LARCFlag_pp (keep = PatientID_OPA Flag_HadLARC_new); 
         set Method_pp&n; 
         retain Flag_HadLARC_new;
         by PatientID_OPA DOSNew;
         if first.PatientID_OPA then Flag_HadLARC_new = 0;
         if MethodLARC = 1 then Flag_HadLARC_new = 1;
         if last.PatientID_OPA then output;
         run;

      %*...FD EDIT === UPDATE Feb 2019 === ;
      %*...FD: add a method flag for each individual method (all include surv codes), and keep them in the final dataset;
      %*...Use retain statement to flag methods;
      data MethodFlag_pp (keep = PatientID_OPA Flag_FemSteril Flag_IUD Flag_Implant Flag_Inject Flag_OC Flag_Patch Flag_Ring Flag_Diaphragm);
         set Method_pp&n;
         retain Flag_FemSteril Flag_IUD Flag_Implant Flag_Inject Flag_OC Flag_Patch Flag_Ring Flag_Diaphragm;
         by PatientID_OPA DOSNew;
         if first.PatientID_OPA then do;
         Flag_FemSteril = 0;
         Flag_IUD = 0;
         Flag_Implant = 0;
         Flag_Inject = 0;
         Flag_OC = 0;
         Flag_Patch = 0;
         Flag_Ring = 0;
         Flag_Diaphragm = 0; end;
         if FemSteril = 1 then Flag_FemSteril = 1;
         if IUD = 1 then Flag_IUD = 1;
         if Implant = 1 then Flag_Implant = 1;
         if Inject = 1 then Flag_Inject = 1;
         if OC = 1 then Flag_OC = 1;
         if Patch = 1 then Flag_Patch = 1;
         if Ring = 1 then Flag_Ring = 1;
         if Diaphragm = 1 then Flag_Diaphragm = 1;
         if last.PatientID_OPA then output;
         run;

      %*...Create MergeMethodLARC dataset by merging Method dataset and LARCFlag dataset by Patient ID;
      data MergeMethodLARC_pp; 
         merge Method_pp&n (in = a) LARCFlag_pp MethodFlag_pp;
         by PatientID_OPA;
         if a = 1;
         run;

      %*...Sort MergeMethodLARC dataset by Patient ID and MethodEffect to create a dataset with the last record for 
           each individual being her most effective method in the measurement year;
      proc sort data = MergeMethodLARC_pp;
         by PatientID_OPA MethodEffect;
         run;

      %*...Create a one record per person dataset with the most effective method in the measurement year variables;
      data ContraceptiveDataMEM_pp (keep = PatientID_OPA MEM_Method_new MEM_MethodEffect_new MEM_MethodMost_new Flag_HadLARC_new 
                                    Flag_FemSteril Flag_IUD Flag_Implant Flag_Inject Flag_OC Flag_Patch Flag_Ring Flag_Diaphragm); 
         set MergeMethodLARC_pp; 
         by PatientID_OPA MethodEffect;
         if last.PatientID_OPA;
         MEM_Method_new = Method;
         MEM_MethodEffect_new = MethodEffect;
         MEM_MethodMost_new = MethodMost;
         run;

      %*...Sort ContraceptiveDataMEM dataset by Patient ID to prepare to merge with MergeMethodLARC dataset to create a
           final dataset with most effective method in the measurement year variables;
      proc sort data = ContraceptiveDataMEM_pp; 
         by PatientID_OPA;
         run;

      %*...Sort by Patient ID to prepare to merge with Method dataset to create final MEM dataset;
      proc sort data = Method_pp&n; 
         by PatientID_OPA;
         run;

      %*...Create MEMLast dataset by merging the MergeMethodLARC dataset with ContraceptiveDataMEM dataet by Patient ID;
      data MEM_pp;  
         merge Method_pp&n
               ContraceptiveDataMEM_pp ;
         by PatientID_OPA;
         run;

      %*...Delete intermediate datasets;
      proc datasets lib = work nolist;
         delete MergeMethodLARC_pp ContraceptiveDataMEM_pp LARCFlag_pp MethodFlag_pp;
         quit;
         run;

      %*...Sort MEM dataset by Patient ID and first date of service to prepare to create a one record per person dataset;
      proc sort data = MEM_pp;
         by PatientID_OPA DOSNew;
         run;

      %*...Create a one record per person dataset from MEM dataset;
      data MEMOne_pp; 
         set MEM_pp; 
         by PatientID_OPA DOSNew;
         if last.PatientID_OPA;
         run;
       

      %*...This is the final dataset for the numerator and will be used in the final calculations; 
      %*...FD: add flag for each method;
      %*...Add postpartum days # suffix to method variables for final measure calculations in postpartum populations;
      data FinalContraceptiveData_pp&n;
        set MEMOne_pp 
             (rename = (
                     MEM_Method_new = MEM_Method_new_&n 
                     MEM_MethodEffect_new = MEM_MethodEffect_new_&n
                     MEM_MethodMost_new = MEM_MethodMost_new_&n
                     Flag_HadLARC_new = Flag_HadLARC_new_&n 
                     Flag_FemSteril = Flag_FemSteril_&n 
                     Flag_IUD = Flag_IUD_&n 
                     Flag_Implant = Flag_Implant_&n 
                     Flag_Inject = Flag_Inject_&n 
                     Flag_OC = Flag_OC_&n 
                     Flag_Patch = Flag_Patch_&n 
                     Flag_Ring = Flag_Ring_&n 
                     Flag_Diaphragm = Flag_Diaphragm_&n 
            ));;

         if MEM_MethodMost_new_&n. ^=1 then MEM_MethodMost_new_&n. = 0; *...MEM_MethodMost_new is Most or Mod Effective Method at End of Timeperiod (1=Yes);
         if Flag_HadLARC_new_&n. = 1 then LARCmeasure_new_&n. = 1;      *...Define LARCMeasure_new (LARC method at end of timeperiod) based on LARC flag;
         else LARCMeasure_new_&n. = 0;
         if MEM_MethodMost_new_&n. ^=1 then MEM_Method_new_&n. = "No method";
         attrib MEM_Method_new_&n. label = "Method of Contraception at &n. days postpartum (NQF Measure)"
                MEM_MethodMost_new_&n. label = "Most or Mod Effective Method at &n. days postpartum (1=Yes) (NQF Measure)"
                LARCMeasure_new_&n. label = "LARC Method at &n. days postpartum (1=Yes) (NQF Measure)"
                Flag_HadLARC_new_&n. label = "Flag for had LARC during &n. days postpartum period (NQF Measure Calc)"
                Flag_FemSteril_&n. label = "Flag for had sterilization during &n. days postpartum (1=Yes) (NQF Measure)"
                Flag_IUD_&n. label = "Flag for had IUD during &n. days postpartum (1=Yes) (NQF Measure)"
                Flag_Implant_&n. label = "Flag for had implant during &n. days postpartum (1=Yes) (NQF Measure)"
                Flag_Inject_&n. label = "Flag for had injectible during &n. days postpartum (1=Yes) (NQF Measure)"
                Flag_OC_&n. label = "Flag for had oral pill during &n. days postpartum (1=Yes) (NQF Measure)"
                Flag_Patch_&n. label = "Flag for had parch during &n. days postpartum (1=Yes) (NQF Measure)"
                Flag_Ring_&n. label = "Flag for had ring during &n. days postpartum (1=Yes) (NQF Measure)"
                Flag_Diaphragm_&n. label = "Flag for had diaphragm during &n. days postpartum (1=Yes) (NQF Measure)"
                ;
         run;

      %*...Delete intermediate datasets;
      proc datasets lib = work nolist;
         delete Method_pp&n MEM_pp MEMOne_pp;
         quit;

      %let i = %eval(&i + 1);
      %end;
   %end;
%mend numerator_pp_new;

%numerator_pp_new;





