/*
________________________________________________________________________________

   02.flag-methods.sas 
   OPA Measure Calculations
   Step 2: Format the data, ready for flagging cases

   Summary  : Create analytic variables and arrays with diagnosis, NDC, and 
   procedure flags.

   Authors  : P. Hastings / phil -at- farharbor -dot- com
            : F. Dong / fei -at- farharbor -dot- com 
			: E. Puga / ella -at- farharbor -dot- com   
            : Based on original code by B. Frederiksen -at- DHHS/OPA
   Version  : v3.01
   Date     : 2020-03-20   
   Revisions: None.  
   History  : 2016-12-19(1.00), 2017-02-22(1.01), 2017-05-17(1.02), 
              2018-04-20(1.03), 2018-08-20(1.03a), 2019-02-26 (2.01)
________________________________________________________________________________

*/

*=============================================================================;
*  FORMAT INPUT DATA
*=============================================================================;

proc format;
   value methodeffectf 0='No effective contraceptive method'
                       1='Inject, OC, Patch, Ring, Diaphragm'
                       2='LARC'
                       3='Sterilization';

   run;

*...Create a claims dataset with contraceptive method flags;
data ContraceptiveClaims;  
   set ContraClaims_setup (keep=
         PatientID_OPA DATE_SVC_SASFMT AGE_CALC AgeGroup  
         &ProcCodes. &DiagCodes. &DrugCodes. &stratifying_var_list. &Patientid.); 

   attrib PatientID_OPA label='Patient ID + suffix for programs'
          DATE_SVC_SASFMT label='Date of Service'
          DOSNew label='Date of Service YYYY-MM-DD'
          AGE_CALC label='Age at Date of Service'
          Month label='Month of Date of Service'
          Year label='Year of Date of Service'
          AgeGroup label='Age Group'
          &Patientid. label='Patient ID (original)'
          &ProcCodes. label='Procedure Code'
          &DiagCodes. label='Diagnosis Code'
          &DrugCodes. label='Drug Code'
          Infecund label='Flag for Infecund'
          Pregnant label='Flag for Pregnant'
          LiveBirth label='Flag for Live Birth'
          NonLiveBirth label='Flag for Non-Live Birth'
          PregEnd label='Flag for Pregnancy End'
          MethodMost label='Flag for a Most Effective Contraceptive Method'
          MethodMod label='Flag for a Moderately Effective Contraceptive Method'
          MethodLARC label='Flag for LARC no Surveillance'
          FemSteril label='Flag for Sterilization'
          IUD label='Flag for IUD'
          Implant label='Flag for Implant'
          Inject label='Flag for Inject'
          OC label='Flag for OC'
          Patch label='Flag for Patch'
          Ring label='Flag for Ring'
          Diaphragm label='Flag for Diaphragm'
          Method label='Contraceptive Method'
          MethodEffect label='Method Effectiveness Scale 0-3';
   format DOSNew DATE_SVC_SASFMT yymmddd10. AGE_CALC 2. methodeffect methodeffectf.;
   DOSNew = DATE_SVC_SASFMT;
   Month = month(DOSNew);
   Year = year(DOSNew);

   
   *...Set up arrays for diagnoses and procedure variables;
   array DiagVar {*} &DiagCodes;
   array ProcVar {*} &ProcCodes;
   array DrugVar {*} &DrugCodes;

   *...Create flags for conditions/methods;
   Infecund = 0;
   Pregnant = 0;
   LiveBirth = 0;
   NonLiveBirth = 0;
   PregEnd = 0;   
   MethodMost = 0;
   MethodMod = 0;
   MethodLARC = 0;
   FemSteril = 0;
   IUD = 0;
   Implant = 0;
   Inject = 0;
   OC = 0;
   Patch = 0;
   Ring = 0;
   Diaphragm = 0;

   *...Set flags based on diagnosis codes;
   do i=1 to dim(DiagVar);
      if DiagVar{i} in (&InfecundDiag) then Infecund = 1; 
      if DiagVar{i} in:(&PregDiag) then Pregnant = 1; 
      if DiagVar{i} in:(&NonLiveBirthDiag) then NonLiveBirth = 1; 
      if DiagVar{i} in (&LiveBirthDiag) then LiveBirth = 1; 
      if DiagVar{i} in (&MostModDiag) then MethodMost = 1; 
      if DiagVar{i} in (&ModEffectDiag) then MethodMod = 1; 
      if DiagVar{i} in (&LARCDiag) then MethodLARC = 1; 
      if DiagVar{i} in (&FemSterilDiag) then FemSteril = 1; 
      if DiagVar{i} in (&IUDDiag) then IUD = 1; 
      if DiagVar{i} in (&ImplantDiag) then Implant = 1; 
      if DiagVar{i} in (&InjectDiag) then Inject = 1;
      if DiagVar{i} in (&OCDiag) then OC = 1; 
      if DiagVar{i} in (&PatchDiag) then Patch = 1; 
      if DiagVar{i} in (&VagRingDiag) then Ring = 1;
      if DiagVar{i} in (&DiaphragmDiag) then Diaphragm = 1;
   end;
    
   *...Set flags based on procedure codes;
   do i=1 to dim(ProcVar);
      if ProcVar{i} in (&InfecundProc) then Infecund = 1;
      if ProcVar{i} in (&PregProc) then Pregnant = 1; 
      if ProcVar{i} in (&NonLiveBirthProc) then NonLiveBirth = 1; 
      if ProcVar{i} in (&LiveBirthProc) then LiveBirth = 1;
      if ProcVar{i} in (&MostModProc) then MethodMost = 1; 
      if ProcVar{i} in (&ModEffectProc) then MethodMod = 1; 
      if ProcVar{i} in (&LARCProc) then MethodLARC = 1; 
      if ProcVar{i} in (&FemSterilProc) then FemSteril = 1; 
      if ProcVar{i} in (&IUDProc) then IUD = 1; 
      if ProcVar{i} in (&ImplantProc) then Implant = 1; 
      if ProcVar{i} in (&InjectProc) then Inject = 1; 
      if ProcVar{i} in (&OCProc) then OC = 1; 
      if ProcVar{i} in (&PatchProc) then Patch = 1; 
      if ProcVar{i} in (&VagRingProc) then Ring = 1; 
      if ProcVar{i} in (&DiaphragmProc) then Diaphragm = 1; 
   end;

   *...Set flags based on drug codes;
   do i=1 to dim(DrugVar);
      if DrugVar{i} in (&MostModDrug) then MethodMost = 1; 
      if DrugVar{i} in (&ModEffectDrug) then MethodMod = 1; 
      if DrugVar{i} in (&LARCDrug) then MethodLARC = 1; 
      if DrugVar{i} in (&FemSterilDrug) then FemSteril = 1;
      if DrugVar{i} in (&IUDDrug) then IUD = 1; 
      if DrugVar{i} in (&ImplantDrug) then Implant = 1; 
      if DrugVar{i} in (&InjectDrug) then Inject = 1; 
      if DrugVar{i} in (&OCDrug) then OC = 1; 
      if DrugVar{i} in (&PatchDrug) then Patch = 1; 
      if DrugVar{i} in (&VagRingDrug) then Ring = 1; 
      if DrugVar{i} in (&DiaphragmDrug) then Diaphragm = 1; 
   end;

   *...Set flag for a pregnancy end;
   if NonLiveBirth = 1 or LiveBirth = 1 then PregEnd = 1; 

   *...Set flags for Method, MethodEffect variables;

   *...NQF-ENDORSED MEASURES...;
   if FemSteril = 1 then Method = "Fem_Steril";
   else if IUD = 1 then Method = "IUD";  *...FD: separate IUD and implant;
   else if Implant = 1 then Method = "Implant";  *...FD: separate IUD and implant;
   else if Inject = 1 then Method = "Injectable";
   else if OC = 1 then Method = "OC";
   else if Patch = 1 then Method = "Patch";
   else if Ring = 1 then Method = "Ring";
   else if Diaphragm = 1 then Method = "Diaphragm";
   *...Order by Method Effectiveness ;
   if Method in ("Fem_Steril") then MethodEffect = 3;
   else if Method in ("IUD" "Implant") then MethodEffect = 2;     
   else if Method in ("Injectable" "OC" "Patch" "Ring" "Diaphragm") then MethodEffect = 1;

   *...DEV: use more detailed method effectiveness category to rank the methods [FD];
   /*
   if Method in ("Fem_Steril") then MethodEffect = 5;
   else if Method in ("IUD" "Implant") then MethodEffect = 4;     
   else if Method in ("Injectable") then MethodEffect = 3;
   else if Method in ("OC" "Patch" "Ring") then MethodEffect = 2;
   else if Method in ("Diaphragm") then MethodEffect = 1;
   else MethodEffect = 0;
   *...END DEV;
   */

   if DOSnew=. then delete; *...NOTE: here we delete any records without a service date (invalid)...;
   drop i;
   run;

*...delete setup data to save disk space...[PH];
proc datasets library=work nolist;
   delete contraclaims_setup;
   quit;


