/*
________________________________________________________________________________

   setup_helper.sas 
   OPA Measure Calculations Setup Helper

   Summary  : Option program to assist with troubleshooting for users.

   Authors  : P. Hastings / phil -at- farharbor -dot- com
   Version  : 1.03a
   Date     : 2018-08-20
   Revisions: none.
   History  : 2018-04-20(1.03)
________________________________________________________________________________

*/

%macro setup_helper_wrapper;
options mprint=0 symbolgen=0;
*...clear the log window..;
DM "log; clear; ";
%symdel datetype dobtype agetype ptidtype ptidlen meandate nobs_inputdata SAS_version version_message1 version_message2 num_women nrows avgclaims 
firstdate lastdate missingDOS dvar dvartype newvarname datemessage1 datemessage2 datemessage3 daterangemessage sas_num_date agemessage1  
agemessage2 agemessage3 firstage lastage agerangemessage nobs_druglength nrows_drug mindrglth maxdrglth drugcode1 
drug_message1 drug_message2 drug_message3 drug_message4 datacheck_message1 datacheck_message2 datacheck_message3 
agechoice datechoice datachoice logquestion datafound nvars_meta adios_message1 adios_message2 diaglist proclist druglist
_savelog _filelog /nowarn;

%global datetype dobtype agetype ptidtype ptidlen meandate nobs_inputdata SAS_version version_message1 version_message2 num_women nrows avgclaims 
firstdate lastdate missingDOS dvar dvartype newvarname datemessage1 datemessage2 datemessage3 daterangemessage sas_num_date agemessage1  
agemessage2 agemessage3 firstage lastage agerangemessage nobs_druglength nrows_drug mindrglth maxdrglth drugcode1 
drugcode1 drug_message1 drug_message2 drug_message3 drug_message4 datacheck_message1 datacheck_message2 
datacheck_message3 datafound nvars_meta adios_message1 adios_message2 logquestion diaglist proclist druglist
_savelog _filelog;

%let SAS_version = &SYSVER;

   %put    ;
   %put    ======================================================================================;
   %put    >>             Starting Setup Helper for OPA Contraceptive Measures....             <<;
   %put    ======================================================================================;
   %put    ;

/*create macro variable telling user whether need to do manual download of lookup tables */
%macro sas_version;
%if %sysevalf(&SAS_version. < 9.4) %then %do;
   %let version_message1= Your SAS version requires that you download the contraceptive lookup tables into your SAS work directory.; 
   %let version_message2= Download https://www.hhs.gov/opa/sites/default/files/contraceptive-lookup-tables and unzip to: %sysfunc(getoption(work));
%end;

%if %sysevalf(&SAS_version. >= 9.4) %then %do;
   %let version_message1= This version of SAS enables the automatic download of the contraceptive;
   %let version_message2= lookup tables from the OPA website.;
%end;

%mend sas_version;
%sas_version;
   %put ;
   %put &version_message1;
   %put &version_message2;
   %put ;

/*...check for existence of data file...*/
%macro dsfinder /minoperator ;
%if %sysfunc(exist(cc.&DSN.,data)) %then 
   %do;
   %let datafound=Y;
   %put ;
   %put Dataset cc.&DSN. confirmed. Now looking for the required variables...;
   %put ;
   %end;
   %else %do;
   %let datafound=N;
   %let adios_message1=WARNING: We could not locate your dataset named cc.&DSN.. Please check;
   %let adios_message2=WARNING: the setup program to ensure that your dataset is named properly.;
   %put ;
   %put NOTE: User specified dataset cc.&DSN. does not seem to exist. Check setup program for correct name and location.;
   %put ;
   %end;
%mend dsfinder;
%dsfinder;
*...exit gracefully with a message if data is not found;
%if &datafound=N %then %goto adios;

/* Create Analytic Working Dataset...and fix variables as needed... */
data ContraClaims_setup;
   set cc.&DSN.;
   run;

%macro varfinder /minoperator ;
      *...get info about the key variables specified in the claims data;
      %if %length(&UseSvcAgeforExclusions)=0 %then %let UseSvcAgeforExclusions=N;
      data meta;
         retain varname type length npos varnum label format informat libname memname memtype ;
         set sashelp.vcolumn (keep=name type length npos varnum label format informat libname memname memtype rename=(name=varname)) ;
         where upcase(libname) = 'WORK' and memtype = 'DATA' and upcase(memname) = "%upcase(ContraClaims_setup)";
         %if %trim(%upcase(&UseSvcAgeforExclusions)) in Y YES %then 
            %do; 
             if trim(upcase(varname)) in 
               ("%upcase(&PatientID)","%upcase(&DateOfService)","%upcase(&AgeAtDateOfService)") 
             then output; 
            %end;
         %else 
            %do; 
             if trim(upcase(varname)) in 
               ("%upcase(&PatientID)","%upcase(&DateOfService)","%upcase(&PatientDOB)") 
             then output;  
            %end;
      run;
      %symdel nvars_meta /nowarn; 
      *...get number of vars found in meta, make sure there are 3;
      data _NULL_;
         if 0 then set meta nobs=n;
         call symputx('nvars_meta',n);
         stop;
      run;
      %if %eval(&nvars_meta) < 3 %then 
         %do;
         %let adios_message1=WARNING: We did not find all required variables for the programs. Please check your setup ;
         %let adios_message2=WARNING: to confirm that PatientID, DateofService, and PatientDOB (or Age) are named properly.;
         %end;
%mend varfinder;
%varfinder;

%if %eval(&nvars_meta) < 3 %then %goto adios;

%macro metavartypes /minoperator;
/* Get variable types for date and age variables */

         data _null_;
         set meta ;
         if trim(upcase(varname))="%upcase(&PatientID)" then 
            do;
            call symput('ptidtype', type);
            call symput('ptidlen', length);
            end;
         if trim(upcase(varname))="%upcase(&DateOfService)" then call symput('datetype', type);
         %if %trim(%upcase(&UseSvcAgeforExclusions)) in Y YES %then 
            %do; 
            if trim(upcase(varname))="%upcase(&AgeatDateOfService)" then call symput('agetype', type);
            %end;
         %else 
            %do; 
            if trim(upcase(varname))="%upcase(&PatientDOB)" then call symput('dobtype', type);
         %end;
         run;
%put ;
%put ============================================================================;
%put PatientID type and length: &ptidtype., &ptidlen. ;
%put DateofService type: &datetype ;
%put Age or DOB type: &dobtype &agetype ;
%put ============================================================================;
%put ;
%mend metavartypes;
%metavartypes;

/* Get number of observations in input data */
%macro number_obs;

%if %sysevalf(&SAS_version. >= 9.0) %then %do;
/* SAS V9.x and higher */
data _NULL_;
   if 0 then set ContraClaims_setup nobs=n;
   call symputx('nrows',n);
   stop;
run;
%let nobs_inputdata=&nrows;
%end;

/* Before SAS V9*/
%else %if %sysevalf(&SAS_version. < 9.0) %then %do;
data _NULL_;
   if 0 then set sashelp.class nobs=n;
   call symput('nrows',trim(left(put(n,8.))));
   stop;
run;
%let nobs_inputdata=&nrows;
%end;

%mend number_obs;
%number_obs;


/* Get number of patients */
proc sort data=ContraClaims_setup;
by &PatientID.;
run;

data patlevel;
set ContraClaims_setup (keep=&PatientID.);
by &PatientID.;
if last.&PatientID.;
run;

%symdel nrows /nowarn;

data _NULL_;
   if 0 then set patlevel nobs=n;
   call symputx('nrows',n);
   stop;
run;

%let num_women=&nrows;

/* Get average number of claims */
data avg_claims;
set ContraClaims_setup (keep=&PatientID.);
by &PatientID.;
retain totclaims;
if first.&PatientID. then totclaims=0;
totclaims+1;
if last.&PatientID then output;
run;

proc means data=avg_claims ndec=2 noprint;
var totclaims;
output out=avgclaimsmean mean=avgclaims;
run;

data _null_;
  set avgclaimsmean;
   call symput("avgclaims",put(avgclaims,8.2));
run;

/* Print variables with the user specified proc, diag, and drug prefixes */

data diag;
set ContraClaims_setup (keep=&DiagCodePrefix:);
run;

data proc;
set ContraClaims_setup (keep=&ProcCodePrefix:);
run;

data drug;
set ContraClaims_setup (keep=&DrugCodePrefix:);
run;

proc contents data=diag out=diagcontents noprint;
run;

proc sql noprint;
select name into :diaglist separated by ", " from diagcontents;

proc contents data=proc out=proccontents noprint;
run;

proc sql noprint;
select name into :proclist separated by ", " from proccontents;

proc contents data=drug out=drugcontents noprint;
run;

proc sql noprint;
select name into :druglist separated by ", " from drugcontents;
quit;

%put ;
%put ============================================================================;
%put Diagnosis Vars: &diaglist ;
%put Procedure Vars: &proclist ;
%put Drug Code Vars: &druglist ;
%put ============================================================================;
%put ;

%if (%cmpres(%length(&diaglist.))<1 or %cmpres(%length(&diaglist.))<1 or %cmpres(%length(&diaglist.))<1) %then 
   %do;
   %let adios_message1=WARNING: One or more of your Diag, Proc, or Drug code vars are missing (see above).;
   %let adios_message2=WARNING: Check the setup to ensure the prefixes and variables exist in your data.;
   %end;
%if (%cmpres(%length(&diaglist.))<1 or %cmpres(%length(&diaglist.))<1 or %cmpres(%length(&diaglist.))<1) %then %goto adios;

%macro newptid;
   /*...This creates a new PatientID variable for analytic purposes, sets new length  */
   /*   Allows for &PatientID as numeric or character, appends placeholder "_0" for delivery episodes */
   data ContraClaims_setup;
      set ContraClaims_setup;
      %if &ptidtype. = num %then 
         %do;
         *Must have a patientID;
         if &PatientID.=. then delete; 
         PatientID_OPA = trim(right(put(&PatientID.,best.0))||"_0");
         %end;
      /* if age is character, AGE_CALC = input character age as numeric */
      %else %if &ptidtype. = char %then 
         %do;
         PatientID_OPA = put(trim(right(&PatientID.)),%cmpres($%cmpres(&ptidlen).))||"_0";
         %end;
      run;
%mend newptid;
%newptid;

/* if Date Variables (Service Date, PatientDOB) are character or unformatted numeric, reformat as a SAS date */
%macro date_num (dvar, dvartype, newvarname);
%let sas_num_date=;
%if &dvartype. = char %then 
   %do;
   data ContraClaims_setup;
   set ContraClaims_setup;
   &newvarname = input(&dvar., anydtdte32.);
   format &newvarname. yymmddd10.;
   run;
   %end;

%else %if &dvartype. = num %then 
   %do;
   proc sql noprint;
   select mean(&dvar.) format=10. into :meandate from ContraClaims_setup;
   quit;
   %* this range # of days from sasdate origin of 1960 would allow roughly DOB years 1932-2200;
   %if %eval(&meandate.<90000) and %eval(-10000<&meandate.) %then %let sas_num_date=yes; 
   %else %let sas_num_date=no;

   %put ;
   %put =========================================================;
   %put Mean date value in your dataset: &meandate.;
   %put &dvar. is numeric in sas date format? &sas_num_date.;
   %put =========================================================;
   %put ;

   %if &sas_num_date. = yes %then 
      %do; 
      data ContraClaims_setup;
      set ContraClaims_setup;
      &newvarname. = &dvar.;
      run;
      %end;

   %else  
      %do;
      data ContraClaims_setup;
      set ContraClaims_setup;
      &newvarname. = input(compress(&dvar.), anydtdte.);
      format &newvarname. yymmddd10.;
      run;
      %end;

%end;

   %if ((&dvartype. = char) or %eval(-10000< &meandate. <90000)) %then 
      %do;
      proc print data = ContraClaims_setup (obs=10);
      format &newvarname. yymmddd10.;
      var &dvar. &newvarname.;
      title2 "Comparing original and converted date variables, first 10 obs (original variable: &dvar.)";
      run;
      title;
      %end;

%mend date_num;

%macro fixdates /minoperator;

   %if %trim(%upcase(&UseSvcAgeforExclusions)) in N NO %then 
      %do; 
      %date_num (&PatientDOB., &dobtype., PT_DOB_SASFMT);
      %let PatientDOB = PT_DOB_SASFMT;
      %end;
   %date_num (&dateofservice., &datetype., DATE_SVC_SASFMT);
   /* get range of dates using either original date variable or new date variable if had to be fixed */
   proc sql noprint;
   select min(DATE_SVC_SASFMT) format=mmddyy10. into :firstdate  from ContraClaims_setup;
   select max(DATE_SVC_SASFMT) format=mmddyy10. into :lastdate   from ContraClaims_setup;
   select nmiss(DATE_SVC_SASFMT) format=8.0     into :missingDOS from ContraClaims_setup;
   quit;

%mend fixdates;
%fixdates;

%macro age_calcs /minoperator;
   proc format;
      value agegroupf
      1 = 'Ages 15 - 20'
      2 = 'Ages 21 - 44'
      3 = 'Ages < 15 (Excluded)'
      4 = 'Ages > 44 (Excluded)'
      . = 'Missing   (Excluded)';
   run;

   data ContraClaims_setup;
   set ContraClaims_setup;
   format AGE_CALC 2. AgeGroup agegroupf.;
   %if %trim(%upcase(&UseSvcAgeforExclusions)) in Y YES %then 
      %do; 
      /*...Alternate method using Age at Date of Service for exclusions... */
      /*   if age is already numeric, AGE_CALC variable = original age variable */
      %if &agetype. = num %then 
         %do;
         AGE_CALC = &ageatdateofservice.;
         %end;
      /* if age is character, AGE_CALC = input character age as numeric */
      %else %if &agetype. = char %then 
         %do;
         AGE_CALC = input(&ageatdateofservice., best32.);
         %end;
      %end;
   %else 
      %do;
      /* Default Method using DOB: Create AGE_CALC using EOY minus PatientDOB */
      AGE_CALC = INT(INTCK('MONTH', &PatientDOB., "31dec&Year."d)/12);
      IF MONTH(&PatientDOB.) = MONTH("31dec&Year."d) THEN AGE_CALC = AGE_CALC -(DAY(&PatientDOB.)>DAY("31dec&Year."d));
      %end;

   /* using new AGE_CALC find number of women inside and outside the 15-44 age range */
   if 15 <= AGE_CALC < 21 then AgeGroup = 1;
   else if 21 <= AGE_CALC <= 44 then AgeGroup = 2;
   else if 1< AGE_CALC < 15 then AgeGroup = 3;
   else if AGE_CALC > 44 then AgeGroup = 4;
   run;

   proc sort data = ContraClaims_setup;
   by &patientid. DATE_SVC_SASFMT;
   run;

   data onewoman;
   set ContraClaims_setup (keep=&patientid. AgeGroup AGE_CALC 
   %if %trim(%upcase(&UseSvcAgeforExclusions)) in Y YES %then 
      %do; 
         &ageatdateofservice.
      %end;
   );
   by &patientid.;
   if last.&patientid.;
   run;
   proc sort data=onewoman; by &patientid.; run;
   %if %trim(%upcase(&UseSvcAgeforExclusions)) in Y YES %then 
      %do; 
      %if &agetype. = char %then 
         %do;
         *...if age needed conversion, then report results;
         proc print data = onewoman (obs=10);
         var &ageatdateofservice. AGE_CALC AgeGroup;
         title2 "Comparing original and converted AGE variables, first 10 obs";
         run;
         title;
         %end;
      %end;

   /*Now merge AGE_CALC and AGE_GROUP to large claims dataset, to ensure one Age and AgeGroup per woman*/
   data ContraClaims_setup;
   merge onewoman (keep=&patientid. AgeGroup AGE_CALC) 
         ContraClaims_setup (drop=AgeGroup AGE_CALC);
   by &patientid.;
   run;

   proc freq data = onewoman;
   format agegroup agegroupf.;
   tables agegroup / missing;
   title2 "Frequency of Age Categories in your Data";
   run;
   title;
%mend age_calcs;
%age_calcs;

/* get range of ages using either original age variable if numeric or new age variable if it was character */
proc sql noprint;
select min(AGE_CALC)  format=2.0 into :firstage from ContraClaims_setup;
select max(AGE_CALC)  format=2.0 into :lastage  from ContraClaims_setup;
quit;

/* Create messages about age and date vars if they are character variables */
%macro age_date_message /minoperator;
   %if %trim(%upcase(&UseSvcAgeforExclusions)) in Y YES %then 
      %do; 
      %if &agetype. = char %then 
         %do;
         %let agemessage1 = Age Format Check: Note, your AGE variable was in alphanumeric format.;
         %let agemessage2 = Creating a new variable (AGE_CALC) in numeric format...done. A few records comparing;
         %let agemessage3 = original and new variables are shown in the results window. Please confirm the results.;
         %let agerangemessage = After conversion your age values range from;
         %end;
      %end;
   %else 
      %do;
      %let agemessage1 = Age Format Check: Passed.;
      %let agemessage2 = Creating variable (AGE_CALC) used for age exclusions...done.; 
      %let agemessage3 = The new variable will be used for measure calculations.;
      %let agerangemessage = Age values in your data range from;
      %end;

   %if &datetype. = char %then %do;
   %let datemessage1 = Date Format Check: Your DATE OF SERVICE variable was in alphanumeric format.;
   %let datemessage2 = Creating analytic variable (DATE_SVC_SASFMT) in SAS DATE format...done. A few records comparing;
   %let datemessage3 = original and new variables are shown in the results window. Please confirm the results.;
   %let daterangemessage = After conversion your date values range from;
   %end;
   %else %if &sas_num_date. = no %then %do;
   %let datemessage1 = Date Format Check: Your DATE OF SERVICE variable was in numeric format;
   %let datemessage2 = Creating a new variable (DATE_SVC_SASFMT) in SASDATE format...done. A few records comparing;
   %let datemessage3 = original and new variables are shown in the results window. Please confirm the results.;
   %let daterangemessage = After conversion your date values range from;
   %end;
   %else %if &sas_num_date. = yes %then %do;
   %let datemessage1 = Date Format Check: Passed.;
   %let datemessage2 = Your DATE OF SERVICE variable is in the correct;
   %let datemessage3 = numeric SAS DATE format.;
   %let daterangemessage = Date values in your data range from;
   %end;

%mend age_date_message;

%age_date_message;

/* check that data has 11 digit drug codes */
data _null_;
set drugcontents;
if _n_ = 1 then do;
   call symput('drugcode1', name);
end;

data drug_check;
set ContraClaims_setup (keep = &drugcode1.);
drugcodelength = length(trim(&drugcode1.));
run;

proc freq data=drug_check noprint;
where drugcodelength ne 1;
tables drugcodelength/out=druglength_freq;
run;

data _NULL_;
   if 0 then set druglength_freq nobs=n;
   call symputx('nrows_drug',n);
   stop;
run;

/* get range of ages using either original age variable if numeric or new age variable if it was character */
proc sql noprint;
select min(drugcodelength)  format=2.0 into :mindrglth from druglength_freq;
select max(drugcodelength)  format=2.0 into :maxdrglth from druglength_freq;
quit;


%let nobs_druglength=&nrows_drug;

%macro druglength_message;
%if (&nobs_druglength. > 1) %then %do;
/*create message about drug variable if have values different than 11 digits */
%let drug_message1 = Drug Code Format Check: Some of your drug codes are shorter ; 
%let drug_message2 = or longer than the 11 digit standard. Measure rates may be impacted. ;
%let drug_message3 = Check the results window for lengths found in your first drug code variable.;
%let drug_message4 = Drug code lengths in your data range from;
%end;
%else %do;
%let drug_message1 = Drug Code Format Check: Passed.; 
%let drug_message2 = Drug codes are correctly formatted in the 11 digit standard.;
%let drug_message3 = No further action needed.;
%let drug_message4 = Drug code lengths in your data range from;
%end;
%mend druglength_message;
%druglength_message;


%macro save_data;

proc sort data = ContraClaims_setup;
by &Patientid. &DateOfService.;
run;

data sample_data;
set ContraClaims_setup (obs = 100);
run;

data sample_data_pat_level;
set sample_data;
by &Patientid.;
if last.&Patientid.;
run;

data sample_data_pat_level (keep = &patientid. newid);
set sample_data_pat_level;
newid = _N_ + 90000 + 100*round(rannor(0));
run;

data cc.OPA_sample_data (drop = &patientid.);
merge sample_data sample_data_pat_level;
by &patientid.;
run;

%mend save_data;

/* define message window */
%window Setup_Helper color=gray

#1 
/ " >> Setup Helper for OPA Contraceptive Measures: Initial Checks (1 of 2) <<" color=blue
/ 
/ " ===== SUMMARY INFORMATION ABOUT YOUR DATA =====" color=blue
/ "           TOTAL CLAIMS: %cmpres(&nobs_inputdata)" 
/ "         TOTAL PATIENTS: %cmpres(&num_women)"
/ " AVG CLAIMS PER PATIENT: %cmpres(&avgclaims)"
/ 
/
/ " ===== CONFIRM YOUR CLAIMS DATA VARIABLES =====" color=blue
/ " Please check that your diagnosis, procedure, and drug code variables are all listed"
/ " below, with no unwanted variables that share the same prefix. If not, review the"
/ " instructions in 00.setup.sas and edit your prefixes or variable names."
/ 
/ " Your diagnosis code prefix was <&diagcodeprefix.>, and we found these variables:" 
/ " --> %upcase(&diaglist.)" color=green
/
/ " Your procedure code prefix was <&proccodeprefix>, and we found these variables:"
/ " --> %upcase(&proclist.)" color=green
/
/ " Your drug code prefix was <&drugcodeprefix>, and we found these variables:" 
/ " --> %upcase(&druglist.)" color=green
/
/
/ " ===== OPTIONAL CHOICES FROM THE SETUP PROGRAM =====" color=blue
/ " Calculate Postpartum Measures? (y/n): %upcase(&postpartum.)"
/ " Exclusions Based on Age at Last Service? (y/n, default=n): %upcase(&UseSvcAgeforExclusions.)"
/ " Additional Report Stratification Variables (if any): &stratifying_var_list."
/
/ " ===== CHECK OF DOWNLOAD CAPABILITY =====" color=blue
/ " Your SAS Version is: &SAS_version."
/ " &version_message1."  
/ " &version_message2."  
/
/ " NOTE: For troubleshooting you can find a zipfile of the log, a deidentified sample, and data contents here:"
/ " --> &data_dir.\OPA_TAfiles.zip"
/
/ " Press [ENTER] to continue." color=blue
;

%macro execute_data_window;

%display Setup_Helper;

%save_data;

%mend;

%execute_data_window;


%macro execute_data_checks;

%if (&nobs_druglength. > 1) or (&agetype. = char) or (&datetype. = char) or (&sas_num_date. = yes) %then 
%do;
%let datacheck_message1 = INITIAL DATA CHECK: One or more variables above were reformatted to enable the measure calculations.;
%let datacheck_message2 = Please confirm that the ranges for these new variables are correct. More info at the OPA website: ;
%let datacheck_message3 = https://www.hhs.gov/opa/performance-measures/claims-data-sas-program-instructions;
%end;
%else %do;
%let datacheck_message1 = INITIAL DATA CHECK: Passed.;
%let datacheck_message2 = Your data appear to meet the program requirements. For more info visit the OPA website:;
%let datacheck_message3 = https://www.hhs.gov/opa/performance-measures/claims-data-sas-program-instructions;
%end;

%window Data_Checks color=gray

#1 
/ " >> Setup Helper for OPA Contraceptive Measures: Variable Format Checks (2 of 2) <<" color=blue
/
/ " &datemessage1."
/ " &datemessage2." 
/ " &datemessage3." 
/ " &daterangemessage.:"
/ " --> &firstdate. - &lastdate. " color=green
/ "     %cmpres(&missingDOS.) records with missing dates will be dropped..." color=green
/
/ " &agemessage1."
/ " &agemessage2." 
/ " &agemessage3." 
/ " &agerangemessage.:"
/ " --> &firstage. - &lastage." color=green
/ 
/ " &drug_message1."
/ " &drug_message2."
/ " &drug_message3."
/ " &drug_message4.:"
/ " --> %cmpres(&mindrglth.) to %cmpres(&maxdrglth.)" color=green
/ 
/ " &datacheck_message1."
/ " &datacheck_message2."
/ " &datacheck_message3."
/
/ " Press [ENTER] to continue." color=blue
;

%display Data_Checks;

%if &nobs_druglength. > 1 %then %do;
   proc freq data=drug_check;
   where drugcodelength ne 1;
   tables drugcodelength;
   title1;
   title2 "Frequency of lengths for your first drug code variable: %cmpres(&drugcode1.)";
   title3;
   run;
   title;
%end;

* Repeat data check windows inside log...;

   %put    ;
   %put    ;
   %put    ;
   %put    ======================================================================================;
   %put    >>>>>>>      Setup Helper for OPA Contraceptive Measures: Initial Checks       <<<<<<<;
   %put    ======================================================================================;
   %put    ;
   %put    ===== SUMMARY INFORMATION ABOUT YOUR DATA ===== ;
   %put              TOTAL CLAIMS: %cmpres(&nobs_inputdata)  ;
   %put            TOTAL PATIENTS: %cmpres(&num_women) ;
   %put    AVG CLAIMS PER PATIENT: %cmpres(&avgclaims) ; 
   %put    ;
   %put    ;
   %put    ===== CONFIRM YOUR CLAIMS DATA VARIABLES ===== ;
   %put    Please check that your diagnosis, procedure, and drug code variables are all listed;
   %put    below, with no unwanted variables that share the same prefix. If not, review the;
   %put    instructions in 00.setup.sas and edit your prefixes or variable names.;
   %put    ;
   %put    Your diagnosis code prefix was <&diagcodeprefix.>, and we found these variables: ;
   %put    --> %upcase(&diaglist.) ;
   %put    ;
   %put    Your procedure code prefix was <&proccodeprefix>, and we found these variables:;
   %put    --> %upcase(&proclist.) ;
   %put    ;
   %put    Your drug code prefix was <&drugcodeprefix>, and we found these variables: ;
   %put    --> %upcase(&druglist.) ;
   %put    ;
   %put    ;
   %put    ===== OPTIONAL CHOICES FROM THE SETUP PROGRAM ===== ;
   %put    Calculate Postpartum Measures? (y/n): %upcase(&postpartum.);
   %put    Exclusions Based on Age at Last Service? (y/n, default=n): %upcase(&UseSvcAgeforExclusions.);
   %put    Additional Report Stratification Variables (if any): &stratifying_var_list.;
   %put    ;
   %put    ===== CHECK OF DOWNLOAD CAPABILITY ===== ;
   %put    Your SAS Version is: &SAS_version.;
   %put    &version_message1.  ;
   %put    &version_message2.  ;
   %put    ;
   %put    NOTE: For troubleshooting you can find a zipfile of the log, a deidentified sample, and data contents here:;
   %put    --> &data_dir.\OPA_TAfiles.zip;
   %put    ;
   %put    ;
   %put    &datemessage1. ;
   %put    &datemessage2. ;
   %put    &datemessage3. ;
   %put    &daterangemessage.:;
   %put    --> &firstdate. - &lastdate. ;
   %put        %cmpres(&missingDOS.) records with missing dates will be dropped ... ;
   %put    ;
   %put    &agemessage1.;
   %put    &agemessage2. ;
   %put    &agemessage3. ;
   %put    &agerangemessage.: ;
   %put    --> &firstage. - &lastage. ;
   %put    ;
   %put    &drug_message1.;
   %put    &drug_message2.;
   %put    &drug_message3.;
   %put    &drug_message4.: ;
   %put    --> %cmpres(&mindrglth.) to %cmpres(&maxdrglth.) ;
   %put    ;
   %put    &datacheck_message1. ;
   %put    &datacheck_message2. ;
   %put    &datacheck_message3. ;
   %put    ;
   %put    ======================================================================================;
   %put    ;
   %put    ;
   %put    ;

%mend;



%execute_data_checks;

proc datasets library=work nolist; 
delete meta avg_claims patlevel sample_data_pat_level druglength_freq avgclaimsmean sample_data onewoman drug_check; 
run; quit;

ODS NORESULTS;
ODS LISTING CLOSE;
ODS RTF FILE="&data_dir.\OPA_data_contents.rtf" ;
proc contents data=ContraClaims_setup; run;
ODS RTF CLOSE;
ODS LISTING;
ODS RESULTS;


/* Creating a ZIP file for troubleshooting with ODS PACKAGE */
ods package(newzip) open nopf;
ods package(newzip) add file="&data_dir.\OPA_prog.log";
ods package(newzip) add file="&data_dir.\OPA_data_contents.rtf";
ods package(newzip) add file="&data_dir.\OPA_sample_data.sas7bdat";
ods package(newzip) publish archive 
  properties(
   archive_name="OPA_TAfiles.zip" 
   archive_path="&data_dir."
   );
ods package(newzip) close;

%let adios_message1=Successfully exited the setup helper. Need more help? ;
%let adios_message2=Contact the OPA for technical assistance: Brittni.Frederiksen@hhs.gov;

%adios: 
proc datasets lib=work nolist; delete diag proc drug diagcontents proccontents drugcontents; run; quit;

%put ;
%put ======================================================================================; 
%put &adios_message1;
%put &adios_message2;
%put ======================================================================================; 
%put ;

%macro savelog;

 data _null_;
 _savelog="&data_dir.\OPA_prog.log";
 _filelog="dm log 'file "!!'"'!!trim(left(_savelog))!!'"'!!" replace' ";
 call symput('_savelog',_savelog);
 call symput('_filelog',_filelog);
 run;

 &_filelog.;

%mend savelog;
%savelog;
options mprint=1;
%mend setup_helper_wrapper;

%setup_helper_wrapper;

