/*
________________________________________________________________________________

   05.report-rates.sas 
   OPA Measure Calculations
   Step 5: Perform final calcuations for measures

   Summary  : Merges together datasets created in 03.denominator.sas and 
              04.numerator.sas programs. Calculates and reports  
              the contraceptive performance measures.

   Authors  : P. Hastings / phil -at- farharbor -dot- com
            : H. Monti / holly -at- farharbor -dot- com  
            : Based on original code by B. Frederiksen -at- DHHS/OPA
   Version  : 1.03a
   Date     : 2018-08-20
   Revisions: Updates to titles for clarity. Added footnote for LARC measure.
              Removed report of all-women measure in postpartum calculation,
              since these require different datasets (due to eligibility.) 
   History  : 2016-12-19(1.00), 2017-02-22(1.01), 2017-05-17(1.02), 
              2018-04-20(1.03)
________________________________________________________________________________

*/

*=============================================================================;
*  Final Calculations of Measure - All Women
*=============================================================================;

*...Sort single record per person dataset from denominator program to prepare to merge with FinalContraceptiveData dataset;
proc sort data = DenominatorDataOne; 
   by PatientID_OPA;
run;

*...Sort final dataset from numerator program to prepare to merge with DenominatorDataOne;
proc sort data = FinalContraceptiveData_all;  
   by PatientID_OPA; 
run;

*...Clear global keep list macro variables;
%symdel DenominatorDataKeepList NumeratorDataKeepList/nowarn;

%global DenominatorDataKeepList NumeratorDataKeepList;

     %let DenominatorDataKeepList = PatientID_OPA  
           AGE_CALC &stratifying_var_list. &Patientid. AgeGroup DOSnew 
           month year exclude_all;
      %let NumeratorDataKeepList = PatientID_OPA MethodMost MethodMod MethodLARC 
            
           Flag_HadLARC_new 
           MEM_Method_new MEM_MethodMost_new ;                                          
           /*MethodEffect FemSteril IUD Implant Inject OC Patch Ring Diaphragm Method 
           MEM_MethodMost_newNS MEM_Method MEM_MethodMost SubMethodIndi Flag_HadLARC 
           Flag_RemovalLARC Flag_ReinsertLARC Flag_InsertLARC Reinsert LARC_Removal 
           LARC_Reinsert LARC_Insert MEM_Method_newNS Flag_HadLARC_newNS */

*...Merge datasets created in numerator and denominator programs by Patient ID;
data FinalContraception;                                                                           
   merge DenominatorDataOne (keep = &DenominatorDataKeepList.)
         FinalContraceptiveData_all (keep = &NumeratorDataKeepList.); 
   by PatientID_OPA;
   if MEM_MethodMost_new ^=1 then MEM_MethodMost_new = 0;
   if Flag_HadLARC_new = 1 then LARCmeasure_new = 1; 
   else LARCmeasure_new = 0;
   if MEM_MethodMost_new ^=1 then MEM_Method_new = "No method";
   attrib MEM_Method_new label = 'Method of Contraception at End of Timeperiod (NQF Measure)'
          MEM_MethodMost_new label = 'Most or Mod Effective Method at End of Timeperiod (1=Yes) (NQF Measure)'
          Flag_HadLARC_new label = 'Flag for had LARC (NQF Measure Calculation)'
          LARCMeasure_new label = 'LARC Method at End of Timeperiod (1=Yes) (NQF Measure)'
          ;
run;


*...Delete intermediate datasets;
proc datasets lib = work nolist;
   delete DenominatorDataOne FinalContraceptiveData_all;
   quit;

*===;
*  Final frequencies/calculation of the measure - All Women;
*===;

*...The variable for measure 1 is MEM_MethodMost and the variable for measure 2 is LARCmeasure;
*...To get the measures by age group sort first by agegroup, then run a proc freq by agegroup (1 = ages 15 to 21, 2 = ages 21 to 45);
*...mem_method gives you a breakdown of contraceptive methods used;
proc sort data = FinalContraception;
   by AgeGroup;
   run;

ods graphics on;
ods listing gpath="&work_temp.";
ods html gpath="&work_temp.";

*...This macro performs Final Measure calcuations for all women and stratified by user-specified variables from Stratifying_var_list;
*...Frequencies of Exclusions, mem_method_new, mem_methodmost_new, LARCMeasure_new, using 2017 specifications;

%macro finalmeasure_stratify;
	%if &postpartum_st. = n %then %do; %*...EP - added so allw report does not show when postpartum is YES;
	%*...EP - moved frequency tables for Exclusions, MEM_Method_new, MEM_methodmost_new, LARCMeasure_new so these 3 allw reports do not print when postpartum is YES;
	proc freq data = FinalContraception;
	   table exclude_all/missing;
	   title1 " ";
	   title2 "== NQF-Endorsed Contraceptive Measures for &year. ==";
	   title3 "Exclusions: Among All Women";
	   title4 "Continuously Eligible or Family Planning Waiver";
	   title5 " "; 
	   run;
	   title; footnote;


	%*...Final Measure calcuations for all women using new LARC method, 2017 specs;
	%*...Frequencies of mem_method_new, mem_methodmost_new, LARCMeasure_new;
	proc freq data = FinalContraception;
	   *...Women with any of the three exclusion criteria are not included in the calculation of the measures;   
	   where Exclude_all = 0;                                                                          
	   table MEM_Method_new  /plots=freqplot (orient=vertical scale=percent type=barchart) ;
	   table MEM_MethodMost_new LARCMeasure_new ;
	   title1 " ";
	   title2 "== NQF-Endorsed Contraceptive Measures for &year. ==";
      title3 "All Women (Ages 15-44)";
	   title4 "Continuously Eligible or Family Planning Waiver";
	   title5 " "; 
      footnote1 "Note: Any LARC is counted (even if later removed during the year). ";
	   run;
	   title; footnote;


	%*...Final Measure calcuations for all women by age group using new LARC method;
	%*...Frequencies of mem_method_new, mem_methodmost_new, LARCMeasure_new;
	proc freq data = FinalContraception;
	   %*...Women that meet any of the 3 exclusion criteria are not included in the calculation of the measures;
	   where Exclude_all = 0;                                                                           
	   by AgeGroup;
	   table MEM_Method_new /plots=freqplot (orient=vertical scale=percent type=barchart);
	   table MEM_MethodMost_new LARCMeasure_new ;
	   title1 " ";
	   title2 "== NQF-Endorsed Contraceptive Measures for &year. ==";
	   title3 "All Women (by Agegroup)";
	   title4 "Continuously Eligible or Family Planning Waiver";
	   title5 " "; 
      footnote1 "Note: Any LARC is counted (even if later removed during the year). ";
	   run;
	   title; footnote;

   	%*...Final Measure calcuations for all women stratified by user-specified variables;
	   %let i = 1;
	   %do %while (%scan(&stratifying_var_list, &i) ne );
	   %let n = %scan(&stratifying_var_list, &i);

	   proc sort data = FinalContraception;
	      by &n.;
	      run;

	   proc freq data = FinalContraception;
	      where Exclude_all = 0;                                                                          %*...Women that meet any of the three exclusion criteria are not included in the
	                                                                                                          calculation of the measures;
	      by &n.;
	      table MEM_Method_new /plots=freqplot (orient=vertical scale=percent type=barchart);
	      table MEM_MethodMost_new LARCMeasure_new ;
	      title1 " ";
   	   title2 "== NQF-Endorsed Contraceptive Measures for &year. ==";
	      title3 "All Women (by &n.)";
	      title4 "Continuously Eligible or Family Planning Waiver";
   	   title5 " "; 
         footnote1 "Note: Any LARC is counted (even if later removed during the year). ";
	      run;
   	   title; footnote;

	   %let i = %eval(&i + 1);
	   %end;
	%end; %*...end postpartum = n;
%mend finalmeasure_stratify;

%finalmeasure_stratify;

*=============================================================================;
*  Final Calculations of Measure - Postpartum Section
*=============================================================================;

%macro ppcalcs_wrapper;
   %if &postpartum_st. = y %then %do;

   *...Clear global keep list macro variables;
   %symdel DenominatorDataKeepList NumeratorDataKeepList/nowarn;

   *...This macro defines the variable keep list for the Numerator and Denominator Data for the postpartum datasets;
   %macro DenomKeep;

   %global DenominatorDataKeepList NumeratorDataKeepList;

         %let DenominatorDataKeepList = PatientID_OPA 
              AGE_CALC &stratifying_var_list. &Patientid. AgeGroup LiveBirthAgeGroup DOSnew 
              month year exclude_all exclude_pp flag_ppindi: flagall_ppindi: ;
         %let NumeratorDataKeepList = PatientID_OPA MethodMost: MethodMod: MethodLARC: 
              /* FemSteril: IUD: Implant: Inject: OC: Patch: Ring: Diaphragm: Method: MethodEffect: Flag_HadLARC:  */
              
              MEM_Method_: MEM_MethodMost: LARCmeasure: ;
   %mend DenomKeep;

   %DenomKeep;

*...This macro merges the denominator and numerator datasets for the postpartum datasets;

%macro ppfinaldata;

   %let i = 1;
   %do %while (%scan(&dayspp_list, &i) ne );
   %let n = %scan(&dayspp_list, &i);

   %*...Sort single record per person dataset from denominator program to prepare to merge with FinalContraceptiveData dataset;
   proc sort data = DenominatorDataOne_pp; 
      by PatientID_OPA;
      run;


   %*...Sort final dataset from numerator program to prepare to merge with DenominatorDataOne;
   proc sort data = FinalContraceptiveData_pp&n;  
      by PatientID_OPA; 
      run;


   %*...Merge datasets created in numerator and denominator programs by Patient ID;
   data FinalContraception_pp&n. ;                                                                           
      merge DenominatorDataOne_pp (keep = &DenominatorDataKeepList.)
            FinalContraceptiveData_pp&n (keep = &NumeratorDataKeepList.); 
      by PatientID_OPA;
      run;

   *...Delete intermediate datasets;
   proc datasets lib = work nolist;
      delete FinalContraceptiveData_pp&n;
      quit;

   %let i = %eval(&i + 1);
   %end;

%mend ppfinaldata;
%ppfinaldata;

proc datasets lib = work nolist;
   delete DenominatorDataOne_pp Numeratordataone_pp Numeratordata_pp;
   quit;



*...The macro performs Final Measure calcuations for postpartum women, stratified by user-specified variables from Stratifying_var_list;
*...Frequencies of mem_method, mem_methodmost, LARCMeasure;
%macro finalmeasure_stratify_pp;

   %let j = 1;
   %do %while (%scan(&stratifying_var_list, &j) ne );
   %let m = %scan(&stratifying_var_list, &j);

   proc sort data=FinalContraception_pp&n.;
      by &m.;
      run;

   proc freq data = FinalContraception_pp&n.;
      *...Only postpartum women aged 15-44 that do not meet either of the two postpartum exclusion criteria are included in the calculation of the measures;
      where flag_ppindi_&n. = 1 and exclude_pp = 0 and LiveBirthAgeGroup in (1,2);                  
      by &m.;
      table MEM_Method_new_&n. /plots=freqplot (orient=vertical scale=percent type=barchart);
      table MEM_MethodMost_new_&n. LARCMeasure_new_&n. ;
      title1 " ";
   	title2 "== NQF-Endorsed Contraceptive Measures for &year. ==";
      title3 "Within &n. Days Postpartum, All Women (by &m.)";
      title4 "Women are continuously enrolled in Medicaid for 60 days postpartum.";
   	title5 " "; 
      footnote1 "Note: Any LARC is counted (even if later removed during the year). ";
      run;
	   title; footnote;

   %let j = %eval(&j + 1);
   %end;
%mend finalmeasure_stratify_pp;

*...This macro performs Final Measure calculations for postpartum women first by age group and then by stratifying variables, if applicable;
*...Frequencies of mem_method, mem_methodmost, LARCMeasure;
%macro ppcalcs/minoperator;

%let i = 1;
   %do %while (%scan(&dayspp_list, &i) ne );
   %let n = %scan(&dayspp_list, &i);
 
   proc freq data = FinalContraception_pp&n.;
      *...use flagall_ppindi_&n. (every claim flagged for pp timeframe) instead of flag_ppindi_&n. (only inclusions) [PH] ;
      where flagall_ppindi_&n. = 1; 
      table exclude_pp /missing;
      title1 " ";
   	title2 "== NQF-Endorsed Contraceptive Measures for &year. ==";
      title3 "Exclusions: Within &n. Days Postpartum, All Women (Ages 15-44)";
      title4 "Women are continuously enrolled in Medicaid for 60 days postpartum.";
   	title5 " "; 
      run;
	   title; footnote;

   proc sort data = FinalContraception_pp&n.;
      by LiveBirthAgeGroup;
      run;

   proc freq data = FinalContraception_pp&n.;
      where flag_ppindi_&n. = 1 & exclude_pp = 0;                                                                                                                        
      %*...Only postpartum women aged 15-44 that do not meet either of the two postpartum exclusion criteria are included in the calculation of the measures;
      table MEM_Method_new_&n. /plots=freqplot (orient=vertical scale=percent type=barchart);
      table MEM_MethodMost_new_&n. LARCMeasure_new_&n. ;
      title1 " ";
   	title2 "== NQF-Endorsed Contraceptive Measures for &year. ==";
      title3 "Within &n. Days Postpartum, Women (Ages 15-44)";
      title4 "Women are continuously enrolled in Medicaid for 60 days postpartum.";
   	title5 " "; 
      footnote1 "Note: Any LARC is counted (even if later removed during the year). ";
      run;
	   title; footnote;

   proc freq data = FinalContraception_pp&n.;
      %*...Only postpartum women aged 15-44 that do not meet either of the two postpartum exclusion criteria are included in the calculation of the measures;
      where flag_ppindi_&n. = 1 & exclude_pp = 0;                                                                                                                        
      by LiveBirthAgeGroup;                                                                        
      table MEM_Method_new_&n. /plots=freqplot (orient=vertical scale=percent type=barchart);
      table MEM_MethodMost_new_&n. LARCMeasure_new_&n. ;
      title1 " ";
   	title2 "== NQF-Endorsed Contraceptive Measures for &year. ==";
      title3 "Within &n. Days Postpartum, Women (by Agegroup)";
      title4 "Women are continuously enrolled in Medicaid for 60 days postpartum.";
   	title5 " "; 
      footnote1 "Note: Any LARC is counted (even if later removed during the year). ";
      run;
	   title; footnote;
   %*...Final measure calculations by stratifying variables;

   %finalmeasure_stratify_pp;

   %let i = %eval(&i + 1);
   %end;

%mend ppcalcs;
%ppcalcs;

%end; %*...end postpartum = y;
%mend ppcalcs_wrapper;
%ppcalcs_wrapper;

ods graphics off;
/*
ods html close;
ods listing;
*/
