/*
________________________________________________________________________________

   04.make-numerator.sas 
   OPA Measure Calculations
   Step 4: Define numerator for measures

   Summary  : Defines numerator using dataset created in 03.denominator.sas
   program, to find patients with LARC and Most/Mod claims. 

   Authors  : P. Hastings / phil -at- farharbor -dot- com
            : H. Monti / holly -at- farharbor -dot- com  
            : Based on original code by B. Frederiksen -at- DHHS/OPA
   Version  : 1.03a
   Date     : 2018-08-20
   Revisions: None. 
   History  : 2016-12-19(1.00), 2017-02-22(1.01), 2017-05-17(1.02), 
              2018-04-20(1.03)
________________________________________________________________________________

*/

*=============================================================================;
*  CREATE NUMERATOR FOR THE MEASURES HERE, MEM/LARC ALGORITHM
*=============================================================================;

*===;
*     RECODE/FLAG VARIOUS LARC CONDITIONS - ALL WOMEN;
*===;

*...This creates a dataset for Methods;
data Method; 
   set NumeratorData; 
   run;   

*...Sort Method data by Patient ID and date of service;
proc sort data = Method; 
   by PatientID_OPA DOSNew; 
   run;

*...This section counts the LARC even if removed, but uses original LARC definition (includes surv codes for IUD/Implant);
*...Use retain statement to flag LARCs [PH];
data LARCFlag (keep = PatientID_OPA Flag_HadLARC_new);
   set Method; 
   retain Flag_HadLARC_new;
   by PatientID_OPA DOSNew;
   if first.PatientID_OPA then Flag_HadLARC_new = 0;
   if MethodLARC = 1 then Flag_HadLARC_new = 1;
   if last.PatientID_OPA then output;
   run;

*...Create MergeMethodLARC dataset by merging Method dataset and LARCFlag dataset by Patient ID;
data MergeMethodLARC (keep = PatientID_OPA DOSNew Method MethodEffect MethodMost Flag_HadLARC_new ); 
   merge method (in=a) LARCFlag ;
   by PatientID_OPA;
   if a=1;
   run;

*...Sort MergeMethodLARC dataset by Patient ID and MethodEffect to create a dataset with the last record for 
    each individual being her most effective method in the measurement year;
proc sort data = MergeMethodLARC;
   by PatientID_OPA MethodEffect;
   run;

*...Create a one record per person dataset with the most effective method in the measurement year variables;
data ContraceptiveDataMEM (keep = PatientID_OPA MEM_Method_new MEM_MethodEffect_new MEM_MethodMost_new Flag_HadLARC_new ); 
   set MergeMethodLARC; 
   by PatientID_OPA MethodEffect;
   if last.PatientID_OPA;
   MEM_Method_new = Method;
   MEM_MethodEffect_new = MethodEffect;
   MEM_MethodMost_new = MethodMost;
   run;

*...Sort ContraceptiveDataMEM dataset by Patient ID to prepare to merge with MergeMethodLARC dataset to 
    create a final dataset with most effective method in the measurement year variables;
proc sort data = ContraceptiveDataMEM; 
   by PatientID_OPA;
   run;

*...Sort by Patient ID to prepare to merge with Method dataset to create final MEM dataset;
proc sort data = Method; 
   by PatientID_OPA;
   run;

*...Create MEMLast dataset by merging the MergeMethodLARC dataset with ContraceptiveDataMEM dataset by Patient ID;
data MEM; 
   merge Method
         ContraceptiveDataMEM ;
   by PatientID_OPA;
   run;

*...Delete intermediate datasets;
proc datasets lib = work nolist;
   delete MergeMethodLARC ContraceptiveDataMEM LARCFlag ;
   quit;
   run;

*...Sort MEM dataset by Patient ID and first date of service to prepare to create a one record per person dataset;
proc sort data = MEM;
   by PatientID_OPA DOSNew;
   run;

*...Create a one record per person dataset from MEM dataset;
data FinalContraceptiveData_all; 
   set MEM; 
   by PatientID_OPA DOSNew;
   if last.PatientID_OPA;
   run;

*...Delete intermediate datasets;
proc datasets lib = work nolist;
   delete Method MEM NumeratorData NumeratorDataOne;
   quit;
   run;

*===============================;
*     POSTPARTUM SECTION        ;
*===============================;

*...This macro performs numerator operations for postpartum populations and creates separate datasets for each 
    postpartum time period (triggered by the postpartum=Y option in 00.setup.sas);

%macro numerator_pp_new;
%if &postpartum_st. = y %then 
   %do;
   %let i = 1;
      %do %while (%scan(&dayspp_list, &i) ne );
      %let n = %scan(&dayspp_list, &i);
      
      %*...This creates a dataset (Method) with only observations where a method could be defined from the codes above making 
           a smaller dataset to work with;
      data Method_pp&n;
         set NumeratorData_pp;
         where flag_pp&n. = 1;
         run;

      %*...Sort Method data by Patient ID and date of service;
      proc sort data = Method_pp&n; 
         by PatientID_OPA DOSNew;
         run;

      %*...This section counts the LARC even if removed, but uses original LARC definition (includes surv codes for IUD/Implant)
           and uses a retain statement to flag LARC;
      data LARCFlag_pp (keep = PatientID_OPA Flag_HadLARC_new); 
         set Method_pp&n; 
         retain Flag_HadLARC_new;
         by PatientID_OPA DOSNew;
         if first.PatientID_OPA then Flag_HadLARC_new = 0;
         if MethodLARC = 1 then Flag_HadLARC_new = 1;
         if last.PatientID_OPA then output;
         run;

      %*...Create MergeMethodLARC dataset by merging Method dataset and LARCFlag dataset by Patient ID;
      data MergeMethodLARC_pp; 
         merge Method_pp&n (in = a) LARCFlag_pp;
         by PatientID_OPA;
         if a = 1;
         run;

      %*...Sort MergeMethodLARC dataset by Patient ID and MethodEffect to create a dataset with the last record for 
           each individual being her most effective method in the measurement year;
      proc sort data = MergeMethodLARC_pp;
         by PatientID_OPA MethodEffect;
         run;

      %*...Create a one record per person dataset with the most effective method in the measurement year variables;
      data ContraceptiveDataMEM_pp (keep = PatientID_OPA MEM_Method_new MEM_MethodEffect_new MEM_MethodMost_new Flag_HadLARC_new); 
         set MergeMethodLARC_pp; 
         by PatientID_OPA MethodEffect;
         if last.PatientID_OPA;
         MEM_Method_new = Method;
         MEM_MethodEffect_new = MethodEffect;
         MEM_MethodMost_new = MethodMost;
         run;

      %*...Sort ContraceptiveDataMEM dataset by Patient ID to prepare to merge with MergeMethodLARC dataset to create a
           final dataset with most effective method in the measurement year variables;
      proc sort data = ContraceptiveDataMEM_pp; 
         by PatientID_OPA;
         run;



      %*...Sort by Patient ID to prepare to merge with Method dataset to create final MEM dataset;
      proc sort data = Method_pp&n; 
         by PatientID_OPA;
         run;

      %*...Create MEMLast dataset by merging the MergeMethodLARC dataset with ContraceptiveDataMEM dataet by Patient ID;
      data MEM_pp;  
         merge Method_pp&n
               ContraceptiveDataMEM_pp ;
         by PatientID_OPA;
         run;

      %*...Delete intermediate datasets;
      proc datasets lib = work nolist;
         delete MergeMethodLARC_pp ContraceptiveDataMEM_pp LARCFlag_pp ;
         quit;
         run;

      %*...Sort MEM dataset by Patient ID and first date of service to prepare to create a one record per person dataset;
      proc sort data = MEM_pp;
         by PatientID_OPA DOSNew;
         run;

      %*...Create a one record per person dataset from MEM dataset;
      data MEMOne_pp; 
         set MEM_pp; 
         by PatientID_OPA DOSNew;
         if last.PatientID_OPA;
         run;


      %*...This is the final dataset for the numerator and will be used in the final calculations; 
      %*...Add postpartum days # suffix to method variables for final measure calculations in postpartum populations;
      data FinalContraceptiveData_pp&n;
        set MEMOne_pp 
             (rename = (
                     MEM_Method_new = MEM_Method_new_&n 
                     MEM_MethodEffect_new = MEM_MethodEffect_new_&n
                     MEM_MethodMost_new = MEM_MethodMost_new_&n
                     Flag_HadLARC_new = Flag_HadLARC_new_&n 
            ));;

         if MEM_MethodMost_new_&n. ^=1 then MEM_MethodMost_new_&n. = 0; *...MEM_MethodMost_new is Most or Mod Effective Method at End of Timeperiod (1=Yes);
         if Flag_HadLARC_new_&n. = 1 then LARCmeasure_new_&n. = 1;      *...Define LARCMeasure_new (LARC method at end of timeperiod) based on LARC flag;
         else LARCMeasure_new_&n. = 0;
         if MEM_MethodMost_new_&n. ^=1 then MEM_Method_new_&n. = "No method";
         attrib MEM_Method_new_&n. label = "Method of Contraception at &n. days postpartum (NQF Measure)"
                MEM_MethodMost_new_&n. label = "Most or Mod Effective Method at &n. days postpartum (1=Yes) (NQF Measure)"
                LARCMeasure_new_&n. label = "LARC Method at &n. days postpartum (1=Yes) (NQF Measure)"
                Flag_HadLARC_new_&n. label = "Flag for had LARC during &n. days postpartum period (NQF Measure Calc)"
                ;
         run;

      %*...Delete intermediate datasets;
      proc datasets lib = work nolist;
         delete Method_pp&n MEM_pp MEMOne_pp;
         quit;

      %let i = %eval(&i + 1);
      %end;
   %end;
%mend numerator_pp_new;

%numerator_pp_new;





